#include "stdafx.h"
#include <stdio.h>
#include <stdarg.h>
#include "RwCommon.h"
#include <rpanim.h>
#include <rpskin.h>
#include <rpspline.h>
#include <rpmorph.h>
#include <rpcollis.h>
#include <rpusrdat.h>
#include <rpprtsys.h>
#include <rphanim.h>
#include <rpmatfx.h>
#include <rtbmp.h>

#ifdef RW_PATCH_EXP

#include <rppatch.h>

#endif /* RW_PATCH_EXP */

#include "RwExpError.h"

int RwLib::m_instances = 0;
BOOL RwLib::m_opened = FALSE;

static void
DebugMessageHandler(RwDebugType type, const RwChar * str)
{
    //Let's have Renderware debug asserts & such printed to Visual C's output window.
    OutputDebugString(str);
    OutputDebugString(RWSTRING("\n"));

    //Also, if this was an assertion, let's attempt to exit gracefully
    //rather than carry on and crash.
    if (type == rwDEBUGASSERT)
    {
        RwExpError err(   std::string("Sorry, couldn't complete export because of a bug!\n")
                        +   std::string("Please contact Renderware technical support with this error message and your MAX file.\n")
                        +   std::string("Renderware assertion failed: ")
                        +   std::string(str) );

        //we display before throwing in this case, because it is possible that max will crash
        //before our exception is caught! The problem is some renderware objects don't get destroyed
        //properly when exceptions are thrown since they're not stack based.
        //We should work towards having stack based wrappers for all our Renderware objects.
        //Until then, at least we give some indication of what's going wrong.
        err.Display();

        throw err;
    }
}

RwLib::RwLib(HWND window, BOOL anim, BOOL skinning, BOOL spline, BOOL morph, BOOL collision, BOOL label, BOOL hanim)
{
    if (m_instances == 0)
    {
        RwEngineOpenParams openParams;

        RwLibPreConstructor();

        RwEngineInit(NULL, 0);

        //we do this to avoid any confusing debug messages
        RwResourcesSetArenaSize(2 * 1024 * 1024);
        //Let's have Renderware debug asserts & such printed to Visual C's output window.
        //Also, if this was an assertion, let's attempt to exit gracefully
        //rather than carry on and crash.
        RwDebugSetHandler(DebugMessageHandler);
        RwDebugSendMessage(rwDEBUGMESSAGE, 0, RWSTRING("Debugging Initialized"));

        if (RpWorldPluginAttach())
        {
            RWEXPMESSAGE(("%s(%d): World module opened successfully!\n",
                         __FILE__, __LINE__));
        }
        else
        {
            RWEXPMESSAGE(("%s(%d): World module failed to open!\n",
                          __FILE__, __LINE__));
            m_opened = FALSE;

            return;
        }

        if(RwLibCustomPlugin())
        {
            RWEXPMESSAGE(("%s(%d): Custom plugin opened successfully!\n",
                          __FILE__, __LINE__));
        }
        else
        {
            RWEXPMESSAGE(("%s(%d): Custom plugin failed to open!\n",
                          __FILE__, __LINE__));
            m_opened = FALSE;

            return;
        }

        if (anim)
        {
            if (RpAnimPluginAttach())
            {
                RWEXPMESSAGE(("%s(%d): Anim plugin opened successfully!\n",
                              __FILE__, __LINE__));
            }
            else
            {
                RWEXPMESSAGE(("%s(%d): Anim plugin failed to open!\n",
                              __FILE__, __LINE__));
                m_opened = FALSE;

                return;
            }
        }

        if (skinning)
        {
            if (RpSkinPluginAttach())
            {
                RWEXPMESSAGE(("%s(%d): Skin plugin opened successfully!\n",
                              __FILE__, __LINE__));
            }
            else
            {
                RWEXPMESSAGE(("%s(%d): Skin plugin failed to open!\n",
                              __FILE__, __LINE__));
                m_opened = FALSE;

                return;
            }
        }

        if (spline)
        {
            if (RpSplinePluginAttach())
            {
                RWEXPMESSAGE(("%s(%d): Spline plugin opened successfully!\n",
                              __FILE__, __LINE__));
            }
            else
            {
                RWEXPMESSAGE(("%s(%d): Spline plugin failed to open!\n",
                              __FILE__, __LINE__));
                m_opened = FALSE;

                return;
            }
        }

        if (morph)
        {
            if (RpMorphPluginAttach())
            {
                RWEXPMESSAGE(("%s(%d): Morph plugin opened successfully!\n",
                              __FILE__, __LINE__));
            }
            else
            {
                RWEXPMESSAGE(("%s(%d): Morph plugin failed to open!\n",
                              __FILE__, __LINE__));
                m_opened = FALSE;

                return;
            }
        }

        if (collision)
        {
            if (RpCollisionPluginAttach())
            {
                RWEXPMESSAGE(("%s(%d): Collision plugin opened successfully!\n",
                              __FILE__, __LINE__));
            }
            else
            {
                RWEXPMESSAGE(("%s(%d): Collision plugin failed to open!\n",
                              __FILE__, __LINE__));
                m_opened = FALSE;

                return;
            }
        }

        if (label)
        {
            if (RpUserDataPluginAttach())
            {
                RWEXPMESSAGE(("%s(%d): Userdata plugin opened successfully!\n",
                              __FILE__, __LINE__));
            }
            else
            {
                RWEXPMESSAGE(("%s(%d): Userdata plugin failed to open!\n",
                              __FILE__, __LINE__));
                m_opened = FALSE;

                return;
            }
        }

        if (RpParticlesAtomicPluginAttach())
        {
            RWEXPMESSAGE(("%s(%d): Particles plugin opened successfully!\n",
                          __FILE__, __LINE__));
        }
        else
        {
            RWEXPMESSAGE(("%s(%d): Particles plugin failed to open!\n",
                          __FILE__, __LINE__));
            m_opened = FALSE;

            return;
        }
        if (RpMatFXPluginAttach())
        {
            RWEXPMESSAGE(("%s(%d): MatFX plugin opened successfully!\n",
                          __FILE__, __LINE__));
        }
        else
        {
            RWEXPMESSAGE(("%s(%d): MatFX plugin failed to open!\n",
                          __FILE__, __LINE__));
            m_opened = FALSE;

            return;
        }
        if (hanim)
        {
            if (RpHAnimPluginAttach())
            {
                RWEXPMESSAGE(("%s(%d): HAnim plugin opened successfully!\n",
                              __FILE__, __LINE__));
            }
            else
            {
                RWEXPMESSAGE(("%s(%d): HAnim plugin failed to open!\n",
                              __FILE__, __LINE__));
                m_opened = FALSE;

                return;
            }
        }

#ifdef RW_PATCH_EXP

        if (RpPatchPluginAttach())
        {
            RWEXPMESSAGE(("%s(%d): Patch module opened successfully!\n",
                         __FILE__, __LINE__));
        }
        else
        {
            RWEXPMESSAGE(("%s(%d): Patch module failed to open!\n",
                          __FILE__, __LINE__));
            m_opened = FALSE;

            return;
        }

#endif /* RW_PATCH_EXP */

        openParams.displayID = (void *)window;
        if (!RwEngineOpen(&openParams))
        {
            RwEngineTerm();
            m_opened = FALSE;

            return;
        }
        RwEngineSetSubSystem(0);
        if (!RwEngineStart())
        {
            RwEngineClose();
            RwEngineTerm();
            m_opened = FALSE;

            return;
        }

        RwLibPostConstructor();

        m_opened = TRUE;
    }

    m_instances++;
}

RwLib::~RwLib(void)
{
    m_instances--;

    if (m_instances == 0)
    {
        RwLibPreDeconstructor();
        RwEngineStop();
        RwEngineClose();
        RwEngineTerm();
        RwLibPostDeconstructor();
        m_opened = FALSE;
    }
}

BOOL
RwLib::Opened(void)
{
    return (m_opened);
}

/*  Camera management   */

RwCamera *
CreateCamera(RwInt32 nWidth, RwInt32 nHeight, RwBool bZBuffer)
{
    RwRaster *rpCameraRaster;
    RwRaster *rpCameraZBufferRaster;
    RwFrame *fpCameraFrame;
    RwCamera *cpCamera;

    /*  create camera components */
    if (cpCamera = RwCameraCreate())
    {
        if (fpCameraFrame = RwFrameCreate())
        {
            rpCameraZBufferRaster = NULL;
            if (!bZBuffer || (rpCameraZBufferRaster = RwRasterCreate(nWidth, nHeight, 0, rwRASTERTYPEZBUFFER)))
            {
                if (rpCameraRaster = RwRasterCreate(nWidth, nHeight, 0, rwRASTERTYPECAMERA))
                {
                    /* link camera components together */
                    RwCameraSetZRaster(cpCamera, rpCameraZBufferRaster);
                    RwCameraSetRaster(cpCamera, rpCameraRaster);
                    RwCameraSetFrame(cpCamera, fpCameraFrame);

                    return cpCamera;
                }
                if (bZBuffer)
                    RwRasterDestroy(rpCameraZBufferRaster);
            }
            RwFrameDestroy(fpCameraFrame);
        }
        RwCameraDestroy(cpCamera);
    }

    return NULL;
}
/*****************************************************************************/

void
DestroyCamera(RwCamera *cpCamera)
{

    if (cpCamera)
    {
        RwRasterDestroy(RwCameraGetRaster(cpCamera));
        if (RwCameraGetZRaster(cpCamera))
            RwRasterDestroy(RwCameraGetZRaster(cpCamera));
        RwFrameDestroy(RwCameraGetFrame(cpCamera));
        RwCameraSetFrame(cpCamera, NULL);
        RwCameraDestroy(cpCamera);
    }
}



/*****************************************************************************/char         *
_rwexpsprintf(const char * format,...)
{
    static char   dberr[512];

    va_list         ap;

    va_start(ap, format);
    _vsnprintf( dberr, sizeof(dberr), format, ap );
    va_end(ap);

    return dberr;
}

